import Foundation

/// Service that integrates survey data with the main life expectancy calculations
class SurveyIntegrationService: ObservableObject {
    private let storageService = SurveyStorageService()
    
    @Published var currentSurveyData: SurveyData?
    @Published var isDataAvailable: Bool = false
    
    init() {
        loadSurveyData()
    }
    
    // MARK: - Data Loading
    
    func loadSurveyData() {
        currentSurveyData = storageService.loadSurveyData()
        isDataAvailable = currentSurveyData?.isCompleted ?? false
    }
    
    func refreshData() {
        loadSurveyData()
    }
    
    // MARK: - Risk Factor Calculations
    
    func calculateRiskMultipliers() -> LifestyleRiskFactors {
        guard let survey = currentSurveyData, survey.isCompleted else {
            return LifestyleRiskFactors.neutral
        }
        
        return LifestyleRiskFactors(
            smokingMultiplier: calculateSmokingRisk(survey),
            vapingMultiplier: calculateVapingRisk(survey),
            alcoholMultiplier: calculateAlcoholRisk(survey),
            dietMultiplier: calculateDietRisk(survey),
            exerciseMultiplier: calculateExerciseMultiplier(survey)
        )
    }
    
    // MARK: - Individual Risk Calculations
    
    private func calculateSmokingRisk(_ survey: SurveyData) -> Double {
        switch survey.smokingStatus {
        case .never:
            return 1.0 // No additional risk
        case .former:
            return 1.15 // Former smokers have some residual risk
        case .current:
            let baseRisk = 1.8 // Current smokers base risk multiplier
            let cigaretteRisk = 1.0 + (Double(survey.cigarettesPerDay) * 0.02) // 2% additional risk per cigarette
            return min(baseRisk * cigaretteRisk, 3.0) // Cap at 3x risk
        }
    }
    
    private func calculateVapingRisk(_ survey: SurveyData) -> Double {
        switch survey.vapingStatus {
        case .never:
            return 1.0
        case .former:
            return 1.05 // Minimal residual risk
        case .current:
            let baseRisk = 1.2 // Lower risk than smoking but not zero
            let puffRisk = 1.0 + (Double(survey.puffsPerDay) * 0.0005) // Minimal additional risk per puff
            return min(baseRisk * puffRisk, 1.5) // Cap at 1.5x risk
        }
    }
    
    private func calculateAlcoholRisk(_ survey: SurveyData) -> Double {
        let units = survey.alcoholUnitsPerWeek
        
        switch units {
        case 0...7:
            return 0.95 // Light drinking may have protective effects
        case 8...14:
            return 1.0 // Moderate drinking, neutral risk
        case 15...21:
            return 1.1 // Slightly increased risk
        case 22...35:
            return 1.3 // Moderate increased risk
        default:
            return 1.6 // High risk for heavy drinking
        }
    }
    
    private func calculateDietRisk(_ survey: SurveyData) -> Double {
        let fastFoodMeals = survey.fastFoodMealsPerWeek
        
        switch fastFoodMeals {
        case 0...1:
            return 0.95 // Excellent diet
        case 2...3:
            return 1.0 // Good diet
        case 4...6:
            return 1.1 // Moderate risk
        case 7...10:
            return 1.25 // Poor diet
        default:
            return 1.4 // Very poor diet
        }
    }
    
    private func calculateExerciseMultiplier(_ survey: SurveyData) -> Double {
        switch survey.exerciseFrequency {
        case .frequent:
            return 0.8 // 20% risk reduction
        case .regular:
            return 0.85 // 15% risk reduction
        case .occasional:
            return 0.95 // 5% risk reduction
        case .rarely:
            return 1.1 // 10% increased risk
        case .none:
            return 1.3 // 30% increased risk
        }
    }
    
    // MARK: - Life Expectancy Impact
    
    func calculateLifeExpectancyAdjustment(baseLifeExpectancy: Double) -> Double {
        let riskFactors = calculateRiskMultipliers()
        let combinedMultiplier = riskFactors.combinedMultiplier
        
        // Convert risk multiplier to life expectancy adjustment
        // Higher risk = lower life expectancy
        let adjustment = baseLifeExpectancy * (1.0 - (combinedMultiplier - 1.0) * 0.1)
        
        return max(adjustment, baseLifeExpectancy * 0.7) // Cap minimum at 70% of base
    }
    
    // MARK: - Health Recommendations
    
    func getPersonalizedRecommendations() -> [HealthRecommendation] {
        guard let survey = currentSurveyData, survey.isCompleted else {
            return [HealthRecommendation.completeSurvey]
        }
        
        var recommendations: [HealthRecommendation] = []
        
        // Smoking recommendations
        if survey.smokingStatus == .current {
            recommendations.append(.quitSmoking)
        }
        
        // Vaping recommendations
        if survey.vapingStatus == .current {
            recommendations.append(.reduceVaping)
        }
        
        // Exercise recommendations
        if survey.exerciseFrequency == .none || survey.exerciseFrequency == .rarely {
            recommendations.append(.increaseExercise)
        }
        
        // Alcohol recommendations
        if survey.alcoholUnitsPerWeek > 21 {
            recommendations.append(.moderateAlcohol)
        }
        
        // Diet recommendations
        if survey.fastFoodMealsPerWeek > 4 {
            recommendations.append(.improveDiet)
        }
        
        return recommendations
    }
}

// MARK: - Supporting Types

struct LifestyleRiskFactors {
    let smokingMultiplier: Double
    let vapingMultiplier: Double
    let alcoholMultiplier: Double
    let dietMultiplier: Double
    let exerciseMultiplier: Double
    
    var combinedMultiplier: Double {
        return smokingMultiplier * vapingMultiplier * alcoholMultiplier * dietMultiplier * exerciseMultiplier
    }
    
    static var neutral: LifestyleRiskFactors {
        return LifestyleRiskFactors(
            smokingMultiplier: 1.0,
            vapingMultiplier: 1.0,
            alcoholMultiplier: 1.0,
            dietMultiplier: 1.0,
            exerciseMultiplier: 1.0
        )
    }
}

enum HealthRecommendation: String, CaseIterable, Identifiable {
    case completeSurvey = "complete_survey"
    case quitSmoking = "quit_smoking"
    case reduceVaping = "reduce_vaping"
    case increaseExercise = "increase_exercise"
    case moderateAlcohol = "moderate_alcohol"
    case improveDiet = "improve_diet"
    
    var id: String { rawValue }
    
    var title: String {
        switch self {
        case .completeSurvey:
            return "Complete Lifestyle Survey"
        case .quitSmoking:
            return "Quit Smoking"
        case .reduceVaping:
            return "Reduce Vaping"
        case .increaseExercise:
            return "Increase Exercise"
        case .moderateAlcohol:
            return "Moderate Alcohol Consumption"
        case .improveDiet:
            return "Improve Diet Quality"
        }
    }
    
    var description: String {
        switch self {
        case .completeSurvey:
            return "Complete the lifestyle survey to get personalized life expectancy predictions."
        case .quitSmoking:
            return "Quitting smoking is the single most important thing you can do for your health."
        case .reduceVaping:
            return "While less harmful than smoking, reducing vaping can improve your health outcomes."
        case .increaseExercise:
            return "Regular exercise can add years to your life and improve quality of life."
        case .moderateAlcohol:
            return "Reducing alcohol intake to recommended levels can improve long-term health."
        case .improveDiet:
            return "Eating fewer processed foods and more whole foods supports longevity."
        }
    }
    
    var priority: Int {
        switch self {
        case .completeSurvey: return 0
        case .quitSmoking: return 1
        case .increaseExercise: return 2
        case .improveDiet: return 3
        case .moderateAlcohol: return 4
        case .reduceVaping: return 5
        }
    }
}